#ifndef __TComplex__
#define __TComplex__

#include "../Basics/CCountedObject.hpp"
using Exponent::Basics::CCountedObject;

//	===========================================================================

namespace Exponent
{
	namespace MathTools
	{
		/**
		 * @class TComplex TComplex.hpp
		 * @brief Complex object
		 *
		 * @date 14/06/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: TComplex.hpp,v 1.3 2007/02/08 21:06:44 paul Exp $
		 */
		template <class TypeName> class TComplex : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 * @param real The real part
			 * @param imag The imag part
			 */
			TComplex(const TypeName real = 0, const TypeName imag = 0) : m_real(real), m_imag(imag)
			{
				EXPONENT_CLASS_CONSTRUCTION(TComplex<TypeName>);
			}

			/**
			 * Destruction
			 */
			virtual ~TComplex()
			{
				EXPONENT_CLASS_DESTRUCTION(TComplex<TypeName>);
			}

//	===========================================================================
			
			/**
			 * Addition operation
			 * @param complex The complex to add this to
			 * @retval TComplex This + complex
			 */
			TComplex<TypeName> operator + (const TComplex<TypeName> &complex)
			{
				TComplex<TypeName> answer(m_real + complex.m_real, m_imag + complex.m_imag);
				return answer;
			}
			
			/**
			 * Subtraction operation
			 * @param complex The complex to subtract this to
			 * @retval TComplex This - complex
			 */
			TComplex<TypeName> operator - (const TComplex<TypeName> &complex)
			{
				TComplex<TypeName> answer(m_real - complex.m_real, m_imag - complex.m_imag);
				return answer;
			}
			
			/**
			 * Multiplication operation
			 * @param complex The complex to times this to
			 * @retval TComplex This * complex
			 */
			TComplex<TypeName> operator * (const TComplex<TypeName> &complex)
			{
				TComplex<TypeName> answer(m_real * complex.m_real - m_imag * complex.m_imag, m_real * complex.m_imag + complex.m_imag * complex.m_real);
				return answer;
			}
			
			/**
			 * Division operation
			 * @param complex The complex to divide this to
			 * @retval TComplex This / complex
			 */
			TComplex<TypeName> operator / (const TComplex<TypeName> &complex)
			{
				TComplex<TypeName> answer(m_real * complex.m_real + m_imag * complex.m_imag, m_imag * complex.m_real - m_real * complex.m_imag);
				return answer / complex.getMagnitudeSquared();
			}
			
			/**
			 * Assignment operation
			 * @param complex The complex to assign this to
			 * @retval TComplex This
			 */
			TComplex<TypeName> operator = (const TComplex<TypeName> &complex)
			{
				TComplex<TypeName> answer(complex.m_real, complex.m_imag);
				return answer;
			}
			
			/**
			 * Multiplication operation
			 * @param value The value to times this to
			 * @retval TComplex This * complex
			 */
			TComplex<TypeName> operator * (const TypeName value)
			{
				TComplex<TypeName> answer(m_real * value, m_imag * value);
				return answer;
			}
			
			/**
			 * Division operation
			 * @param value The value to divide this to
			 * @retval TComplex This / value
			 */
			TComplex<TypeName> operator / (const TypeName value)
			{
				assert(value != 0);
				TComplex<TypeName> answer(m_real / value, m_imag / value);
				return answer;
			}

//	===========================================================================

			/**
			 * Set the real and imaginary from a magnitude and phase components
			 * @param magnitude The magnitude part
			 * @param phase The phase part
			 */
			void setRealAndImaginaryFromMagnitudeAndPhase(const TypeName magnitude, const TypeName phase)
			{	
				m_real = magnitude * cos(phase);
				m_imag = magnitude * sin(phase);
			}

			/**
			 * Set real and imaginary parts
			 * @param real The real part
			 * @param imag The imag part
			 */
			void setRealAndImaginary(const TypeName real, const TypeName imag)
			{
				m_real = real;
				m_imag = imag;
			}

//	===========================================================================

			/**
			 * Get the magnitude
			 * @retval TypeName The magnitude (as defined sqrt(m_real^2+m_imag^2))
			 */
			TypeName getMagnitude() const
			{
				return sqrt(m_real * m_real + m_imag * m_imag);
			}

			/**
			 * Get magnitude squared
			 * @retval TypeName Magnitude^2
			 */
			TypeName getMagnitudeSquared() const
			{
				return m_real * m_real + m_imag * m_imag;
			}

			/**
			 * Get the phase
			 * @retval TypeName The phase (as defined atan2(m_imag, m_real))
			 */
			TypeName getPhase() const
			{
				return atan2(m_imag, m_real);
			}

//	===========================================================================

			/**
			 * Get the real part
			 * @retval TypeName The real part
			 */
			TypeName getReal() const
			{
				return m_real;
			}

			/**
			 * Get the imaginary part
			 * @retval TypeName The imaginary part
			 */
			TypeName getImaginary() const
			{
				return m_imag;
			}

//	===========================================================================

		protected:

//	===========================================================================

			TypeName m_real;			/**< real value */
			TypeName m_imag;			/**< imaginary component */
		};

		/** @cond */
		EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(TComplex<TypeName>, TypeName, CCountedObject);
		/** @endcond */
	}
}
#endif	// End of TComplex.hpp